/******************************************************************************

 @file  simple_peripheral.c

 @brief This file contains the Simple BLE Peripheral sample application for use
        with the CC2650 Bluetooth Low Energy Protocol Stack.

 Group: WCS, BTS
 Target Device: CC2650, CC2640, CC1350

 ******************************************************************************

 Copyright (c) 2013-2016, Texas Instruments Incorporated
 All rights reserved.

 Redistribution and use in source and binary forms, with or without
 modification, are permitted provided that the following conditions
 are met:

 *  Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.

 *  Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in the
    documentation and/or other materials provided with the distribution.

 *  Neither the name of Texas Instruments Incorporated nor the names of
    its contributors may be used to endorse or promote products derived
    from this software without specific prior written permission.

 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

 ******************************************************************************
 Generated by:
 BDS version: 1.1.3135.0
 Plugin:      Texas Instruments BLE SDK GATT Server plugin 1.0.8
 Time:        Sat Jun 24 2017 16:55:14 GMT+01:00
 *****************************************************************************/


/*********************************************************************
 * INCLUDES
 */
#include <string.h>

#include <ti/sysbios/knl/Task.h>
#include <ti/sysbios/knl/Clock.h>
#include <ti/sysbios/knl/Semaphore.h>
#include <ti/sysbios/knl/Queue.h>

//#define xdc_runtime_Log_DISABLE_ALL 1  // Add to disable logs from this file
#include <xdc/runtime/Log.h>
#include <xdc/runtime/Diags.h>

#include "hci_tl.h"
#include "gatt.h"
#include "linkdb.h"
#include "gapgattserver.h"
#include "gattservapp.h"
#include "devinfoservice.h"

// Bluetooth Developer Studio services
#include "led_service.h"
#include "button_service.h"
#include "data_service.h"
#include "relay_service.h"
#include "digital_input_service.h"

#if defined(FEATURE_OAD) || defined(IMAGE_INVALIDATE)
#include "oad_target.h"
#include "oad.h"
#endif //FEATURE_OAD || IMAGE_INVALIDATE

#include "peripheral.h"
#include "gapbondmgr.h"

#include "osal_snv.h"
#include "icall_apimsg.h"

#include "util.h"

#ifdef USE_RCOSC
#include "rcosc_calibration.h"
#endif //USE_RCOSC

#include <ti/mw/display/Display.h>
#include "board_key.h"

#include "board.h"

#include "simple_peripheral.h"


/*********************************************************************
 * CONSTANTS
 */

// Advertising interval when device is discoverable (units of 625us, 160=100ms)
#define DEFAULT_ADVERTISING_INTERVAL          160

// Limited discoverable mode advertises for 30.72s, and then stops
// General discoverable mode advertises indefinitely
#define DEFAULT_DISCOVERABLE_MODE             GAP_ADTYPE_FLAGS_GENERAL

#ifndef FEATURE_OAD
// Minimum connection interval (units of 1.25ms, 80=100ms) if automatic
// parameter update request is enabled
#define DEFAULT_DESIRED_MIN_CONN_INTERVAL     80

// Maximum connection interval (units of 1.25ms, 800=1000ms) if automatic
// parameter update request is enabled
#define DEFAULT_DESIRED_MAX_CONN_INTERVAL     800
#else //!FEATURE_OAD
// Minimum connection interval (units of 1.25ms, 8=10ms) if automatic
// parameter update request is enabled
#define DEFAULT_DESIRED_MIN_CONN_INTERVAL     8

// Maximum connection interval (units of 1.25ms, 8=10ms) if automatic
// parameter update request is enabled
#define DEFAULT_DESIRED_MAX_CONN_INTERVAL     8
#endif // FEATURE_OAD

// Slave latency to use if automatic parameter update request is enabled
#define DEFAULT_DESIRED_SLAVE_LATENCY         0

// Supervision timeout value (units of 10ms, 1000=10s) if automatic parameter
// update request is enabled
#define DEFAULT_DESIRED_CONN_TIMEOUT          1000

// Whether to enable automatic parameter update request when a connection is
// formed
#define DEFAULT_ENABLE_UPDATE_REQUEST         TRUE

// Connection Pause Peripheral time value (in seconds)
#define DEFAULT_CONN_PAUSE_PERIPHERAL         6

// How often to perform periodic event (in msec)
#define SBP_PERIODIC_EVT_PERIOD               5000

#ifdef FEATURE_OAD
// The size of an OAD packet.
#define OAD_PACKET_SIZE                       ((OAD_BLOCK_SIZE) + 2)
#endif // FEATURE_OAD

// Task configuration
#define SBP_TASK_PRIORITY                     1


#ifndef SBP_TASK_STACK_SIZE
#define SBP_TASK_STACK_SIZE                   644
#endif

// Internal Events for RTOS application
#define SBP_STATE_CHANGE_EVT                  0x0001
#define SBP_CHAR_CHANGE_EVT                   0x0002
#define SBP_PERIODIC_EVT                      0x0004
#define SBP_CONN_EVT_END_EVT                  0x0008

/*********************************************************************
 * TYPEDEFS
 */

// Types of messages that can be sent to the application task
typedef enum
{
  APP_MSG_SERVICE_WRITE = 0,
  APP_MSG_SERVICE_CFG,
  APP_MSG_SEND_NOTIFICATION,
  APP_MSG_GAP_STATE_CHANGE
} app_msg_types_t;

// Struct for messages sent to the application task
typedef struct
{
  Queue_Elem       _elem;
  app_msg_types_t  type;
  uint8_t          pdu[];
} app_msg_t;

// Struct for messages about characteristic data
typedef struct
{
  uint16_t svcUUID; // UUID of the service
  uint16_t dataLen; //
  uint8_t  paramID; // Index of the characteristic
  uint8_t  data[];  // Flexible array member, extended to malloc - sizeof(.)
} char_data_t;


/*********************************************************************
 * GLOBAL VARIABLES
 */

// Display Interface
Display_Handle dispHandle = NULL;

/*********************************************************************
 * LOCAL VARIABLES
 */

// Entity ID globally used to check for source and/or destination of messages
static ICall_EntityID selfEntity;

// Semaphore globally used to post events to the application thread
static ICall_Semaphore sem;

// Queue object used for application messages.
static Queue_Struct applicationMsgQ;
static Queue_Handle hApplicationMsgQ;

// Clock structs for periodic notification example
static Clock_Struct bs_BUTTON0_clock;
static Clock_Struct bs_BUTTON1_clock;
static Clock_Struct bs_BUTTON2_T1_clock;
static Clock_Struct bs_BUTTON3_T2_clock;
static Clock_Struct ds_Stream_clock;
static Clock_Struct dis_IN1_clock;
static Clock_Struct dis_IN2_clock;
static Clock_Struct dis_IN3_clock;



#if defined(FEATURE_OAD)
// Event data from OAD profile.
static Queue_Struct oadQ;
static Queue_Handle hOadQ;
#endif //FEATURE_OAD

// Task configuration
Task_Struct sbpTask;
Char sbpTaskStack[SBP_TASK_STACK_SIZE];

// Profile state and parameters
//static gaprole_States_t gapProfileState = GAPROLE_INIT;

// GAP - SCAN RSP data (max size = 31 bytes)
static uint8_t scanRspData[] =
{
  // No scan response data provided.
  0x00 // Placeholder to keep the compiler happy.
};

// GAP - Advertisement data (max size = 31 bytes, though this is
// best kept short to conserve power while advertisting)
static uint8_t advertData[] =
{
  // Flags; this sets the device to use limited discoverable
  // mode (advertises for 30 seconds at a time) or general
  // discoverable mode (advertises indefinitely), depending
  // on the DEFAULT_DISCOVERY_MODE define.
  0x02,   // length of this data
  GAP_ADTYPE_FLAGS,
  DEFAULT_DISCOVERABLE_MODE | GAP_ADTYPE_FLAGS_BREDR_NOT_SUPPORTED,

  // complete name
  11,
  GAP_ADTYPE_LOCAL_NAME_COMPLETE,
  'P', 'r', 'o', 'j', 'e', 'c', 't', 'O', 'n', 'e',

};


// GAP GATT Attributes
static uint8_t attDeviceName[GAP_DEVICE_NAME_LEN] = "ProjectOne";

// Globals used for ATT Response retransmission
static gattMsgEvent_t *pAttRsp = NULL;
static uint8_t rspTxRetry = 0;

/*********************************************************************
 * LOCAL FUNCTIONS
 */

static void SimpleBLEPeripheral_init( void );
static void SimpleBLEPeripheral_taskFxn(UArg a0, UArg a1);

static uint8_t SimpleBLEPeripheral_processStackMsg(ICall_Hdr *pMsg);
static uint8_t SimpleBLEPeripheral_processGATTMsg(gattMsgEvent_t *pMsg);

static void SimpleBLEPeripheral_sendAttRsp(void);
static void SimpleBLEPeripheral_freeAttRsp(uint8_t status);

#ifdef FEATURE_OAD
void SimpleBLEPeripheral_processOadWriteCB(uint8_t event, uint16_t connHandle,
                                           uint8_t *pData);
#endif //FEATURE_OAD

static void user_processGapStateChangeEvt(gaprole_States_t newState);
static void user_gapStateChangeCB(gaprole_States_t newState);

// Generic callback handlers for value changes in services.
static void user_service_ValueChangeCB( uint16_t connHandle, uint16_t svcUuid, uint8_t paramID, uint8_t *pValue, uint16_t len );
static void user_service_CfgChangeCB( uint16_t connHandle, uint16_t svcUuid, uint8_t paramID, uint8_t *pValue, uint16_t len );

// Task context handlers for generated services.
static void user_LedService_ValueChangeDispatchHandler(char_data_t *pCharData);

static void user_ButtonService_CfgChangeDispatchHandler(char_data_t *pCharData);

static void user_DataService_ValueChangeDispatchHandler(char_data_t *pCharData);
static void user_DataService_CfgChangeDispatchHandler(char_data_t *pCharData);

static void user_RelayService_ValueChangeDispatchHandler(char_data_t *pCharData);

static void user_DigitalInputService_CfgChangeDispatchHandler(char_data_t *pCharData);

// Callback handler(s) for the clock object(s) used to demonstrate notifications
static void user_ButtonService_clockSwiHandler(UArg paramID);
static void user_DataService_clockSwiHandler(UArg paramID);
static void user_DigitalInputService_clockSwiHandler(UArg paramID);

// Task handler for sending notifications.
static void user_updateCharVal(char_data_t *pCharData);

// Utility functions
static void user_enqueueRawAppMsg( app_msg_types_t appMsgType, uint8_t *pData, uint16_t len );
static void user_enqueueCharDataMsg( app_msg_types_t appMsgType, uint16_t connHandle,
                                       uint16_t serviceUUID, uint8_t paramID,
                                       uint8_t *pValue, uint16_t len );

char *Util_convertArrayToHexString( uint8_t *arr, uint8_t len );

/*********************************************************************
 * PROFILE CALLBACKS
 */

// GAP Role Callbacks
static gapRolesCBs_t user_gapRoleCBs =
{
  user_gapStateChangeCB     // Profile State Change Callbacks
};

// GAP Bond Manager Callbacks
static gapBondCBs_t user_bondMgrCBs =
{
  NULL, // Passcode callback (not used by application)
  NULL  // Pairing / Bonding state Callback (not used by application)
};

#ifdef FEATURE_OAD
static oadTargetCBs_t simpleBLEPeripheral_oadCBs =
{
  SimpleBLEPeripheral_processOadWriteCB // Write Callback.
};
#endif //FEATURE_OAD

// Service callback structure for registering our handlers with the service
// LED Service callback handler. The type LED_ServiceCBs_t is defined in LED_Service.h
static LedServiceCBs_t user_LED_ServiceCBs =
{
  .pfnChangeCb    = user_service_ValueChangeCB, // Characteristic value change callback handler
  .pfnCfgChangeCb = NULL, // No notification-/indication enabled chars in LED Service
};

// Button Service callback handler. The type Button_ServiceCBs_t is defined in Button_Service.h
static ButtonServiceCBs_t user_Button_ServiceCBs =
{
  .pfnChangeCb    = NULL, // No writable chars in Button Service, so no change handler.
  .pfnCfgChangeCb = user_service_CfgChangeCB, // Noti/ind configuration callback handler
};

// Data Service callback handler. The type Data_ServiceCBs_t is defined in Data_Service.h
static DataServiceCBs_t user_Data_ServiceCBs =
{
  .pfnChangeCb    = user_service_ValueChangeCB, // Characteristic value change callback handler
  .pfnCfgChangeCb = user_service_CfgChangeCB, // Noti/ind configuration callback handler
};

// RELAY SERVICE callback handler. The type RELAY_SERVICECBs_t is defined in RELAY_SERVICE.h
static RelayServiceCBs_t user_RELAY_SERVICECBs =
{
  .pfnChangeCb    = user_service_ValueChangeCB, // Characteristic value change callback handler
  .pfnCfgChangeCb = NULL, // No notification-/indication enabled chars in RELAY SERVICE
};

// Digital Input Service callback handler. The type Digital_Input_ServiceCBs_t is defined in Digital_Input_Service.h
static DigitalInputServiceCBs_t user_Digital_Input_ServiceCBs =
{
  .pfnChangeCb    = NULL, // No writable chars in Digital Input Service, so no change handler.
  .pfnCfgChangeCb = user_service_CfgChangeCB, // Noti/ind configuration callback handler
};


/*********************************************************************
 * PUBLIC FUNCTIONS
 */

/*********************************************************************
 * @fn      SimpleBLEPeripheral_createTask
 *
 * @brief   Task creation function for the Simple BLE Peripheral.
 *
 * @param   None.
 *
 * @return  None.
 */
void SimpleBLEPeripheral_createTask(void)
{
  Task_Params taskParams;

  // Configure task
  Task_Params_init(&taskParams);
  taskParams.stack = sbpTaskStack;
  taskParams.stackSize = SBP_TASK_STACK_SIZE;
  taskParams.priority = SBP_TASK_PRIORITY;

  Task_construct(&sbpTask, SimpleBLEPeripheral_taskFxn, &taskParams, NULL);
}

/*********************************************************************
 * @fn      SimpleBLEPeripheral_init
 *
 * @brief   Called during initialization and contains application
 *          specific initialization (ie. hardware initialization/setup,
 *          table initialization, power up notification, etc), and
 *          profile initialization/setup.
 *
 * @param   None.
 *
 * @return  None.
 */
static void SimpleBLEPeripheral_init(void)
{
  // ******************************************************************
  // N0 STACK API CALLS CAN OCCUR BEFORE THIS CALL TO ICall_registerApp
  // ******************************************************************
  // Register the current thread as an ICall dispatcher application
  // so that the application can send and receive messages.
  ICall_registerApp(&selfEntity, &sem);

#ifdef USE_RCOSC
  RCOSC_enableCalibration();
#endif // USE_RCOSC

  // Initialize queue for application messages.
  // Note: Used to transfer control to application thread
  Queue_construct(&applicationMsgQ, NULL);
  hApplicationMsgQ = Queue_handle(&applicationMsgQ);

  Clock_Params clockParams;
  Clock_Params_init(&clockParams);
  clockParams.period = 5000 * (1000 / Clock_tickPeriod); // 5000ms period

  // Clock struct initialization for periodic notification example
  // Clock callbacks only have one parameter, so make one callback handler per service
  // and one Clock Struct per noti/ind characteristic.
  clockParams.arg = BS_BUTTON0_ID;
  Clock_construct(&bs_BUTTON0_clock,
                  user_ButtonService_clockSwiHandler,
                  0,
                  &clockParams);

  clockParams.arg = BS_BUTTON1_ID;
  Clock_construct(&bs_BUTTON1_clock,
                  user_ButtonService_clockSwiHandler,
                  0,
                  &clockParams);

  clockParams.arg = BS_BUTTON2_T1_ID;
  Clock_construct(&bs_BUTTON2_T1_clock,
                  user_ButtonService_clockSwiHandler,
                  0,
                  &clockParams);

  clockParams.arg = BS_BUTTON3_T2_ID;
  Clock_construct(&bs_BUTTON3_T2_clock,
                  user_ButtonService_clockSwiHandler,
                  0,
                  &clockParams);

  clockParams.arg = DS_STREAM_ID;
  Clock_construct(&ds_Stream_clock,
                  user_DataService_clockSwiHandler,
                  0,
                  &clockParams);

  clockParams.arg = DIS_IN1_ID;
  Clock_construct(&dis_IN1_clock,
                  user_DigitalInputService_clockSwiHandler,
                  0,
                  &clockParams);

  clockParams.arg = DIS_IN2_ID;
  Clock_construct(&dis_IN2_clock,
                  user_DigitalInputService_clockSwiHandler,
                  0,
                  &clockParams);

  clockParams.arg = DIS_IN3_ID;
  Clock_construct(&dis_IN3_clock,
                  user_DigitalInputService_clockSwiHandler,
                  0,
                  &clockParams);


  dispHandle = Display_open(Display_Type_LCD, NULL);

  // Setup the GAP
  GAP_SetParamValue(TGAP_CONN_PAUSE_PERIPHERAL, DEFAULT_CONN_PAUSE_PERIPHERAL);

  // Setup the GAP Peripheral Role Profile
  {
    // For all hardware platforms, device starts advertising upon initialization
    uint8_t initialAdvertEnable = TRUE;

    // By setting this to zero, the device will go into the waiting state after
    // being discoverable for 30.72 second, and will not being advertising again
    // until the enabler is set back to TRUE
    uint16_t advertOffTime = 0;

    uint8_t enableUpdateRequest = DEFAULT_ENABLE_UPDATE_REQUEST;
    uint16_t desiredMinInterval = DEFAULT_DESIRED_MIN_CONN_INTERVAL;
    uint16_t desiredMaxInterval = DEFAULT_DESIRED_MAX_CONN_INTERVAL;
    uint16_t desiredSlaveLatency = DEFAULT_DESIRED_SLAVE_LATENCY;
    uint16_t desiredConnTimeout = DEFAULT_DESIRED_CONN_TIMEOUT;

    // Set the GAP Role Parameters
    GAPRole_SetParameter(GAPROLE_ADVERT_ENABLED, sizeof(uint8_t),
                         &initialAdvertEnable);
    GAPRole_SetParameter(GAPROLE_ADVERT_OFF_TIME, sizeof(uint16_t),
                         &advertOffTime);

    GAPRole_SetParameter(GAPROLE_SCAN_RSP_DATA, sizeof(scanRspData),
                         scanRspData);
    GAPRole_SetParameter(GAPROLE_ADVERT_DATA, sizeof(advertData), advertData);

    GAPRole_SetParameter(GAPROLE_PARAM_UPDATE_ENABLE, sizeof(uint8_t),
                         &enableUpdateRequest);
    GAPRole_SetParameter(GAPROLE_MIN_CONN_INTERVAL, sizeof(uint16_t),
                         &desiredMinInterval);
    GAPRole_SetParameter(GAPROLE_MAX_CONN_INTERVAL, sizeof(uint16_t),
                         &desiredMaxInterval);
    GAPRole_SetParameter(GAPROLE_SLAVE_LATENCY, sizeof(uint16_t),
                         &desiredSlaveLatency);
    GAPRole_SetParameter(GAPROLE_TIMEOUT_MULTIPLIER, sizeof(uint16_t),
                         &desiredConnTimeout);
  }

  // Set the GAP Characteristics
  GGS_SetParameter(GGS_DEVICE_NAME_ATT, GAP_DEVICE_NAME_LEN, attDeviceName);

  // Set advertising interval
  {
    uint16_t advInt = DEFAULT_ADVERTISING_INTERVAL;

    GAP_SetParamValue(TGAP_LIM_DISC_ADV_INT_MIN, advInt);
    GAP_SetParamValue(TGAP_LIM_DISC_ADV_INT_MAX, advInt);
    GAP_SetParamValue(TGAP_GEN_DISC_ADV_INT_MIN, advInt);
    GAP_SetParamValue(TGAP_GEN_DISC_ADV_INT_MAX, advInt);
  }

  // Setup the GAP Bond Manager
  {
    uint32_t passkey = 0; // passkey "000000"
    uint8_t pairMode = GAPBOND_PAIRING_MODE_WAIT_FOR_REQ;
    uint8_t mitm = TRUE;
    uint8_t ioCap = GAPBOND_IO_CAP_DISPLAY_ONLY;
    uint8_t bonding = TRUE;

    GAPBondMgr_SetParameter(GAPBOND_DEFAULT_PASSCODE, sizeof(uint32_t),
                            &passkey);
    GAPBondMgr_SetParameter(GAPBOND_PAIRING_MODE, sizeof(uint8_t), &pairMode);
    GAPBondMgr_SetParameter(GAPBOND_MITM_PROTECTION, sizeof(uint8_t), &mitm);
    GAPBondMgr_SetParameter(GAPBOND_IO_CAPABILITIES, sizeof(uint8_t), &ioCap);
    GAPBondMgr_SetParameter(GAPBOND_BONDING_ENABLED, sizeof(uint8_t), &bonding);
  }

   // Initialize GATT attributes
  GGS_AddService(GATT_ALL_SERVICES);           // GAP
  GATTServApp_AddService(GATT_ALL_SERVICES);   // GATT attributes
  DevInfo_AddService();                        // Device Information Service

  // Add generated services to GATT server
  LedService_AddService( selfEntity );
  ButtonService_AddService( selfEntity );
  DataService_AddService( selfEntity );
  RelayService_AddService( selfEntity );
  DigitalInputService_AddService( selfEntity );

  // Register callbacks with the generated services that
  // can generate events (writes received) to the application
  LedService_RegisterAppCBs( &user_LED_ServiceCBs );
  ButtonService_RegisterAppCBs( &user_Button_ServiceCBs );
  DataService_RegisterAppCBs( &user_Data_ServiceCBs );
  RelayService_RegisterAppCBs( &user_RELAY_SERVICECBs );
  DigitalInputService_RegisterAppCBs( &user_Digital_Input_ServiceCBs );
  // Placeholder variable for characteristic intialization
  uint8_t someVal[20] = {0};

  // Initalization of characteristics in LED_Service that can provide data.
  LedService_SetParameter(LS_LED0_ID, LS_LED0_LEN, &someVal);
  LedService_SetParameter(LS_LED1_ID, LS_LED1_LEN, &someVal);

  // Initalization of characteristics in Button_Service that can provide data.
  ButtonService_SetParameter(BS_BUTTON0_ID, BS_BUTTON0_LEN, &someVal);
  ButtonService_SetParameter(BS_BUTTON1_ID, BS_BUTTON1_LEN, &someVal);
  ButtonService_SetParameter(BS_BUTTON2_T1_ID, BS_BUTTON2_T1_LEN, &someVal);
  ButtonService_SetParameter(BS_BUTTON3_T2_ID, BS_BUTTON3_T2_LEN, &someVal);

  // Initalization of characteristics in Data_Service that can provide data.
  DataService_SetParameter(DS_STRING_ID, DS_STRING_LEN, &someVal);
  DataService_SetParameter(DS_STREAM_ID, DS_STREAM_LEN, &someVal);

  // Initalization of characteristics in RELAY_SERVICE that can provide data.
  RelayService_SetParameter(RS_RELAY1_ID, RS_RELAY1_LEN, &someVal);
  RelayService_SetParameter(RS_RELAY2_ID, RS_RELAY2_LEN, &someVal);

  // Initalization of characteristics in Digital_Input_Service that can provide data.
  DigitalInputService_SetParameter(DIS_IN1_ID, DIS_IN1_LEN, &someVal);
  DigitalInputService_SetParameter(DIS_IN2_ID, DIS_IN2_LEN, &someVal);
  DigitalInputService_SetParameter(DIS_IN3_ID, DIS_IN3_LEN, &someVal);


#ifdef FEATURE_OAD
  VOID OAD_addService();                 // OAD Profile
  OAD_register((oadTargetCBs_t *)&simpleBLEPeripheral_oadCBs);
  hOadQ = Util_constructQueue(&oadQ);
#endif //FEATURE_OAD

#ifdef IMAGE_INVALIDATE
  Reset_addService();
#endif //IMAGE_INVALIDATE


  // Start the Device
  VOID GAPRole_StartDevice(&user_gapRoleCBs);

  // Start Bond Manager
  VOID GAPBondMgr_Register(&user_bondMgrCBs);

  // Register with GAP for HCI/Host messages
  GAP_RegisterForMsgs(selfEntity);

  // Register for GATT local events and ATT Responses pending for transmission
  GATT_RegisterForMsgs(selfEntity);

  HCI_LE_ReadMaxDataLenCmd();

#if defined FEATURE_OAD
#if defined (HAL_IMAGE_A)
  Display_print0(dispHandle, 0, 0, "ProjectOne A");
#else
  Display_print0(dispHandle, 0, 0, "ProjectOne B");
#endif // HAL_IMAGE_A
#else
  Display_print0(dispHandle, 0, 0, "ProjectOne");
#endif // FEATURE_OAD
}

/*********************************************************************
 * @fn      SimpleBLEPeripheral_taskFxn
 *
 * @brief   Application task entry point for the Simple BLE Peripheral.
 *
 * @param   a0, a1 - not used.
 *
 * @return  None.
 */
static void SimpleBLEPeripheral_taskFxn(UArg a0, UArg a1)
{
  // Initialize application
  SimpleBLEPeripheral_init();

  // Application main loop
  for (;;)
  {
    // Waits for a signal to the semaphore associated with the calling thread.
    // Note that the semaphore associated with a thread is signaled when a
    // message is queued to the message receive queue of the thread or when
    // ICall_signal() function is called onto the semaphore.
    ICall_Errno errno = ICall_wait(ICALL_TIMEOUT_FOREVER);

    if (errno == ICALL_ERRNO_SUCCESS)
    {
      ICall_EntityID dest;
      ICall_ServiceEnum src;
      ICall_HciExtEvt *pMsg = NULL;

      if (ICall_fetchServiceMsg(&src, &dest,
                                (void **)&pMsg) == ICALL_ERRNO_SUCCESS)
      {
        uint8 safeToDealloc = TRUE;

        if ((src == ICALL_SERVICE_CLASS_BLE) && (dest == selfEntity))
        {
          ICall_Stack_Event *pEvt = (ICall_Stack_Event *)pMsg;

          // Check for BLE stack events first
          if (pEvt->signature == 0xffff)
          {
            if (pEvt->event_flag & SBP_CONN_EVT_END_EVT)
            {
              // Try to retransmit pending ATT Response (if any)
              SimpleBLEPeripheral_sendAttRsp();
            }
          }
          else
          {
            // Process inter-task message
            safeToDealloc = SimpleBLEPeripheral_processStackMsg((ICall_Hdr *)pMsg);
          }
        }

        if (pMsg && safeToDealloc)
        {
          ICall_freeMsg(pMsg);
        }
      }

      // Process messages sent from another task or another context.
      while (!Queue_empty(hApplicationMsgQ))
      {
        app_msg_t   *pMsg = Queue_dequeue(hApplicationMsgQ);
        char_data_t *pCharData = (char_data_t *)pMsg->pdu;

        switch (pMsg->type)
        {
          case APP_MSG_SERVICE_WRITE: /* Message about received value write */
            /* Call different handler per service */
            switch(pCharData->svcUUID) {
              case LED_SERVICE_SERV_UUID:
                user_LedService_ValueChangeDispatchHandler(pCharData);
                break;
              case DATA_SERVICE_SERV_UUID:
                user_DataService_ValueChangeDispatchHandler(pCharData);
                break;
              case RELAY_SERVICE_SERV_UUID:
                user_RelayService_ValueChangeDispatchHandler(pCharData);
                break;
            }
            break;

          case APP_MSG_SERVICE_CFG: /* Message about received CCCD write */
            /* Call different handler per service */
            switch(pCharData->svcUUID) {
              case BUTTON_SERVICE_SERV_UUID:
                user_ButtonService_CfgChangeDispatchHandler(pCharData);
                break;
              case DATA_SERVICE_SERV_UUID:
                user_DataService_CfgChangeDispatchHandler(pCharData);
                break;
              case DIGITAL_INPUT_SERVICE_SERV_UUID:
                user_DigitalInputService_CfgChangeDispatchHandler(pCharData);
                break;
            }
            break;

          case APP_MSG_SEND_NOTIFICATION: /* Message to self from ClockSwi */
            user_updateCharVal(pCharData);
            break;

          case APP_MSG_GAP_STATE_CHANGE:
            user_processGapStateChangeEvt( *(gaprole_States_t *)pMsg->pdu );
            break;
        }

        // Free the received message.
        ICall_free(pMsg);
      }
    }

#ifdef FEATURE_OAD
    while (!Queue_empty(hOadQ))
    {
      oadTargetWrite_t *oadWriteEvt = Queue_dequeue(hOadQ);

      // Identify new image.
      if (oadWriteEvt->event == OAD_WRITE_IDENTIFY_REQ)
      {
        OAD_imgIdentifyWrite(oadWriteEvt->connHandle, oadWriteEvt->pData);
      }
      // Write a next block request.
      else if (oadWriteEvt->event == OAD_WRITE_BLOCK_REQ)
      {
        OAD_imgBlockWrite(oadWriteEvt->connHandle, oadWriteEvt->pData);
      }

      // Free buffer.
      ICall_free(oadWriteEvt);
    }
#endif //FEATURE_OAD
  }
}

/*********************************************************************
 * @fn      SimpleBLEPeripheral_processStackMsg
 *
 * @brief   Process an incoming stack message.
 *
 * @param   pMsg - message to process
 *
 * @return  TRUE if safe to deallocate incoming message, FALSE otherwise.
 */
static uint8_t SimpleBLEPeripheral_processStackMsg(ICall_Hdr *pMsg)
{
  uint8_t safeToDealloc = TRUE;

  switch (pMsg->event)
  {
    case GATT_MSG_EVENT:
      // Process GATT message
      safeToDealloc = SimpleBLEPeripheral_processGATTMsg((gattMsgEvent_t *)pMsg);
      break;

    case HCI_GAP_EVENT_EVENT:
      {
        // Process HCI message
        switch(pMsg->status)
        {
          case HCI_COMMAND_COMPLETE_EVENT_CODE:
            // Process HCI Command Complete Event
            break;

          default:
            break;
        }
      }
      break;

    default:
      // do nothing
      break;
  }

  return (safeToDealloc);
}

/*********************************************************************
 * @fn      SimpleBLEPeripheral_processGATTMsg
 *
 * @brief   Process GATT messages and events.
 *
 * @return  TRUE if safe to deallocate incoming message, FALSE otherwise.
 */
static uint8_t SimpleBLEPeripheral_processGATTMsg(gattMsgEvent_t *pMsg)
{
  // See if GATT server was unable to transmit an ATT response
  if (pMsg->hdr.status == blePending)
  {
    // No HCI buffer was available. Let's try to retransmit the response
    // on the next connection event.
    if (HCI_EXT_ConnEventNoticeCmd(pMsg->connHandle, selfEntity,
                                   SBP_CONN_EVT_END_EVT) == SUCCESS)
    {
      // First free any pending response
      SimpleBLEPeripheral_freeAttRsp(FAILURE);

      // Hold on to the response message for retransmission
      pAttRsp = pMsg;

      // Don't free the response message yet
      return (FALSE);
    }
  }
  else if (pMsg->method == ATT_FLOW_CTRL_VIOLATED_EVENT)
  {
    // ATT request-response or indication-confirmation flow control is
    // violated. All subsequent ATT requests or indications will be dropped.
    // The app is informed in case it wants to drop the connection.

    // Display the opcode of the message that caused the violation.
    Display_print1(dispHandle, 5, 0, "FC Violated: %d", pMsg->msg.flowCtrlEvt.opcode);
  }
  else if (pMsg->method == ATT_MTU_UPDATED_EVENT)
  {
    // MTU size updated
    Display_print1(dispHandle, 5, 0, "MTU Size: $d", pMsg->msg.mtuEvt.MTU);
  }

  // Free message payload. Needed only for ATT Protocol messages
  GATT_bm_free(&pMsg->msg, pMsg->method);

  // It's safe to free the incoming message
  return (TRUE);
}

/*********************************************************************
 * @fn      SimpleBLEPeripheral_sendAttRsp
 *
 * @brief   Send a pending ATT response message.
 *
 * @param   none
 *
 * @return  none
 */
static void SimpleBLEPeripheral_sendAttRsp(void)
{
  // See if there's a pending ATT Response to be transmitted
  if (pAttRsp != NULL)
  {
    uint8_t status;

    // Increment retransmission count
    rspTxRetry++;

    // Try to retransmit ATT response till either we're successful or
    // the ATT Client times out (after 30s) and drops the connection.
    status = GATT_SendRsp(pAttRsp->connHandle, pAttRsp->method, &(pAttRsp->msg));
    if ((status != blePending) && (status != MSG_BUFFER_NOT_AVAIL))
    {
      // Disable connection event end notice
      HCI_EXT_ConnEventNoticeCmd(pAttRsp->connHandle, selfEntity, 0);

      // We're done with the response message
      SimpleBLEPeripheral_freeAttRsp(status);
    }
    else
    {
      // Continue retrying
      Display_print1(dispHandle, 5, 0, "Rsp send retry: %d", rspTxRetry);
    }
  }
}

/*********************************************************************
 * @fn      SimpleBLEPeripheral_freeAttRsp
 *
 * @brief   Free ATT response message.
 *
 * @param   status - response transmit status
 *
 * @return  none
 */
static void SimpleBLEPeripheral_freeAttRsp(uint8_t status)
{
  // See if there's a pending ATT response message
  if (pAttRsp != NULL)
  {
    // See if the response was sent out successfully
    if (status == SUCCESS)
    {
      Display_print1(dispHandle, 5, 0, "Rsp sent retry: %d", rspTxRetry);
    }
    else
    {
      // Free response payload
      GATT_bm_free(&pAttRsp->msg, pAttRsp->method);

      Display_print1(dispHandle, 5, 0, "Rsp retry failed: %d", rspTxRetry);
    }

    // Free response message
    ICall_freeMsg(pAttRsp);

    // Reset our globals
    pAttRsp = NULL;
    rspTxRetry = 0;
  }
}


/*********************************************************************
 * @fn      SimpleBLEPeripheral_processStateChangeEvt
 *
 * @brief   Process a pending GAP Role state change event.
 *
 * @param   newState - new state
 *
 * @return  None.
 */
static void user_processGapStateChangeEvt(gaprole_States_t newState)
{
#ifdef PLUS_BROADCASTER
  static bool firstConnFlag = false;
#endif // PLUS_BROADCASTER

  switch ( newState )
  {
    case GAPROLE_STARTED:
      {
        uint8_t ownAddress[B_ADDR_LEN];
        uint8_t systemId[DEVINFO_SYSTEM_ID_LEN];

        GAPRole_GetParameter(GAPROLE_BD_ADDR, ownAddress);

        // use 6 bytes of device address for 8 bytes of system ID value
        systemId[0] = ownAddress[0];
        systemId[1] = ownAddress[1];
        systemId[2] = ownAddress[2];

        // set middle bytes to zero
        systemId[4] = 0x00;
        systemId[3] = 0x00;

        // shift three bytes up
        systemId[7] = ownAddress[5];
        systemId[6] = ownAddress[4];
        systemId[5] = ownAddress[3];

        DevInfo_SetParameter(DEVINFO_SYSTEM_ID, DEVINFO_SYSTEM_ID_LEN, systemId);

        // Display device address
        Display_print0(dispHandle, 1, 0, Util_convertBdAddr2Str(ownAddress));
        Display_print0(dispHandle, 2, 0, "Initialized");
      }
      break;

    case GAPROLE_ADVERTISING:
      Display_print0(dispHandle, 2, 0, "Advertising");
      break;

#ifdef PLUS_BROADCASTER
    /* After a connection is dropped a device in PLUS_BROADCASTER will continue
     * sending non-connectable advertisements and shall sending this change of
     * state to the application.  These are then disabled here so that sending
     * connectable advertisements can resume.
     */
    case GAPROLE_ADVERTISING_NONCONN:
      {
        uint8_t advertEnabled = FALSE;

        // Disable non-connectable advertising.
        GAPRole_SetParameter(GAPROLE_ADV_NONCONN_ENABLED, sizeof(uint8_t),
                           &advertEnabled);

        advertEnabled = TRUE;

        // Enabled connectable advertising.
        GAPRole_SetParameter(GAPROLE_ADVERT_ENABLED, sizeof(uint8_t),
                             &advertEnabled);

        // Reset flag for next connection.
        firstConnFlag = false;

        SimpleBLEPeripheral_freeAttRsp(bleNotConnected);
      }
      break;
#endif //PLUS_BROADCASTER

    case GAPROLE_CONNECTED:
      {
        linkDBInfo_t linkInfo;
        uint8_t numActive = 0;

        numActive = linkDB_NumActive();

        // Use numActive to determine the connection handle of the last
        // connection
        if ( linkDB_GetInfo( numActive - 1, &linkInfo ) == SUCCESS )
        {
          Display_print1(dispHandle, 2, 0, "Num Conns: %d", (uint16_t)numActive);
          Display_print0(dispHandle, 3, 0, Util_convertBdAddr2Str(linkInfo.addr));
        }
        else
        {
          uint8_t peerAddress[B_ADDR_LEN];

          GAPRole_GetParameter(GAPROLE_CONN_BD_ADDR, peerAddress);

          Display_print0(dispHandle, 2, 0, "Connected");
          Display_print0(dispHandle, 3, 0, Util_convertBdAddr2Str(peerAddress));
        }

        #ifdef PLUS_BROADCASTER
          // Only turn advertising on for this state when we first connect
          // otherwise, when we go from connected_advertising back to this state
          // we will be turning advertising back on.
          if (firstConnFlag == false)
          {
            uint8_t advertEnabled = FALSE; // Turn on Advertising

            // Disable connectable advertising.
            GAPRole_SetParameter(GAPROLE_ADVERT_ENABLED, sizeof(uint8_t),
                                 &advertEnabled);

            // Set to true for non-connectabel advertising.
            advertEnabled = TRUE;

            // Enable non-connectable advertising.
            GAPRole_SetParameter(GAPROLE_ADV_NONCONN_ENABLED, sizeof(uint8_t),
                                 &advertEnabled);
            firstConnFlag = true;
          }
        #endif // PLUS_BROADCASTER
      }
      break;

    case GAPROLE_CONNECTED_ADV:
      Display_print0(dispHandle, 2, 0, "Connected Advertising");
      break;

    case GAPROLE_WAITING:
      SimpleBLEPeripheral_freeAttRsp(bleNotConnected);

      Display_print0(dispHandle, 2, 0, "Disconnected");

      // Clear remaining lines
      Display_clearLines(dispHandle, 3, 7);

      // Turn off periodic clocks for ind/noti demo
      Clock_stop((Clock_Handle)&bs_BUTTON0_clock);
      Clock_stop((Clock_Handle)&bs_BUTTON1_clock);
      Clock_stop((Clock_Handle)&bs_BUTTON2_T1_clock);
      Clock_stop((Clock_Handle)&bs_BUTTON3_T2_clock);
      Clock_stop((Clock_Handle)&ds_Stream_clock);
      Clock_stop((Clock_Handle)&dis_IN1_clock);
      Clock_stop((Clock_Handle)&dis_IN2_clock);
      Clock_stop((Clock_Handle)&dis_IN3_clock);
      break;

    case GAPROLE_WAITING_AFTER_TIMEOUT:
      SimpleBLEPeripheral_freeAttRsp(bleNotConnected);

      Display_print0(dispHandle, 2, 0, "Timed Out");

      // Clear remaining lines
      Display_clearLines(dispHandle, 3, 7);

      #ifdef PLUS_BROADCASTER
        // Reset flag for next connection.
        firstConnFlag = false;
      #endif //#ifdef (PLUS_BROADCASTER)

      // Turn off periodic clocks for ind/noti demo
      Clock_stop((Clock_Handle)&bs_BUTTON0_clock);
      Clock_stop((Clock_Handle)&bs_BUTTON1_clock);
      Clock_stop((Clock_Handle)&bs_BUTTON2_T1_clock);
      Clock_stop((Clock_Handle)&bs_BUTTON3_T2_clock);
      Clock_stop((Clock_Handle)&ds_Stream_clock);
      Clock_stop((Clock_Handle)&dis_IN1_clock);
      Clock_stop((Clock_Handle)&dis_IN2_clock);
      Clock_stop((Clock_Handle)&dis_IN3_clock);
      break;

    case GAPROLE_ERROR:
      Display_print0(dispHandle, 2, 0, "Error");
      break;

    default:
      Display_clearLine(dispHandle, 2);
      break;
  }

  // Update the state
  //gapProfileState = newState;
}


/*********************************************************************
 * @fn      user_LedService_ValueChangeDispatchHandler
 *
 * @brief   Handle in user task context a write request sent as message
 *          to the application task.
 *
 * @param   pCharData  pointer to malloc'd char write data
 *
 * @return  None.
 */
void user_LedService_ValueChangeDispatchHandler(char_data_t *pCharData)
{
  char *data;
  data = Util_convertArrayToHexString(pCharData->data, pCharData->dataLen);

  Display_clearLine(dispHandle, 4);
  Display_print0(dispHandle, 5, 0, "LED Service");
  Display_print0(dispHandle, 7, 0, data);

  switch (pCharData->paramID)
  {
    case LS_LED0_ID:
      // Do something useful with pCharData->data here
      // -------------------------
      Display_print0(dispHandle, 6, 0, "LED0");
      Log_info3("Value Change msg: %s %s: %s",
                (IArg)"LED Service",
                (IArg)"LED0",
                (IArg)data);
      break;

    case LS_LED1_ID:
      // Do something useful with pCharData->data here
      // -------------------------
      Display_print0(dispHandle, 6, 0, "LED1");
      Log_info3("Value Change msg: %s %s: %s",
                (IArg)"LED Service",
                (IArg)"LED1",
                (IArg)data);
      break;

  default:
    return;
  }
}

/*********************************************************************
 * @fn      user_ButtonService_CfgChangeDispatchHandler
 *
 * @brief   Handle in user task context a CCCD write request sent as
 *          message to the application task.
 *
 * @param   pCharData  pointer to malloc'd char write data
 *
 * @return  None.
 */
void user_ButtonService_CfgChangeDispatchHandler(char_data_t *pCharData)
{
  // Cast received data to uint16, as that's the format for CCCD writes.
  uint16_t configValue = *(uint16_t *)pCharData->data;
  char *configValString;

  // Determine what to tell the user
  switch(configValue)
  {
  case GATT_CFG_NO_OPERATION:
    configValString = "Noti/Ind disabled";
    break;
  case GATT_CLIENT_CFG_NOTIFY:
    configValString = "Notifications enabled";
    break;
  case GATT_CLIENT_CFG_INDICATE:
    configValString = "Indications enabled";
    break;
  }

  Display_clearLine(dispHandle, 4);
  Display_print0(dispHandle, 5, 0, "Button Service");
  Display_print0(dispHandle, 7, 0, configValString);

  switch (pCharData->paramID)
  {
    case BS_BUTTON0_ID:
      // Do something useful with configValue here
      // -------------------------
      Display_print0(dispHandle, 6, 0, "BUTTON0");
      Log_info3("CCCD Change msg: %s %s: %s",
                (IArg)"Button Service",
                (IArg)"BUTTON0",
                (IArg)configValString);

      if (configValue) // 0x0001 and 0x0002 both indicate turned on.
        Clock_start((Clock_Handle)&bs_BUTTON0_clock);
      else
        Clock_stop((Clock_Handle)&bs_BUTTON0_clock);
      break;
    case BS_BUTTON1_ID:
      // Do something useful with configValue here
      // -------------------------
      Display_print0(dispHandle, 6, 0, "BUTTON1");
      Log_info3("CCCD Change msg: %s %s: %s",
                (IArg)"Button Service",
                (IArg)"BUTTON1",
                (IArg)configValString);

      if (configValue) // 0x0001 and 0x0002 both indicate turned on.
        Clock_start((Clock_Handle)&bs_BUTTON1_clock);
      else
        Clock_stop((Clock_Handle)&bs_BUTTON1_clock);
      break;
    case BS_BUTTON2_T1_ID:
      // Do something useful with configValue here
      // -------------------------
      Display_print0(dispHandle, 6, 0, "BUTTON2_T1");
      Log_info3("CCCD Change msg: %s %s: %s",
                (IArg)"Button Service",
                (IArg)"BUTTON2_T1",
                (IArg)configValString);

      if (configValue) // 0x0001 and 0x0002 both indicate turned on.
        Clock_start((Clock_Handle)&bs_BUTTON2_T1_clock);
      else
        Clock_stop((Clock_Handle)&bs_BUTTON2_T1_clock);
      break;
    case BS_BUTTON3_T2_ID:
      // Do something useful with configValue here
      // -------------------------
      Display_print0(dispHandle, 6, 0, "BUTTON3_T2");
      Log_info3("CCCD Change msg: %s %s: %s",
                (IArg)"Button Service",
                (IArg)"BUTTON3_T2",
                (IArg)configValString);

      if (configValue) // 0x0001 and 0x0002 both indicate turned on.
        Clock_start((Clock_Handle)&bs_BUTTON3_T2_clock);
      else
        Clock_stop((Clock_Handle)&bs_BUTTON3_T2_clock);
      break;
  }
}

/*********************************************************************
 * @fn      user_DataService_ValueChangeDispatchHandler
 *
 * @brief   Handle in user task context a write request sent as message
 *          to the application task.
 *
 * @param   pCharData  pointer to malloc'd char write data
 *
 * @return  None.
 */
void user_DataService_ValueChangeDispatchHandler(char_data_t *pCharData)
{
  char *data;
  data = Util_convertArrayToHexString(pCharData->data, pCharData->dataLen);

  Display_clearLine(dispHandle, 4);
  Display_print0(dispHandle, 5, 0, "Data Service");
  Display_print0(dispHandle, 7, 0, data);

  switch (pCharData->paramID)
  {
    case DS_STRING_ID:
      // Do something useful with pCharData->data here
      // -------------------------
      Display_print0(dispHandle, 6, 0, "String");
      Log_info3("Value Change msg: %s %s: %s",
                (IArg)"Data Service",
                (IArg)"String",
                (IArg)data);
      break;

    case DS_STREAM_ID:
      // Do something useful with pCharData->data here
      // -------------------------
      Display_print0(dispHandle, 6, 0, "Stream");
      Log_info3("Value Change msg: %s %s: %s",
                (IArg)"Data Service",
                (IArg)"Stream",
                (IArg)data);
      break;

  default:
    return;
  }
}

/*********************************************************************
 * @fn      user_DataService_CfgChangeDispatchHandler
 *
 * @brief   Handle in user task context a CCCD write request sent as
 *          message to the application task.
 *
 * @param   pCharData  pointer to malloc'd char write data
 *
 * @return  None.
 */
void user_DataService_CfgChangeDispatchHandler(char_data_t *pCharData)
{
  // Cast received data to uint16, as that's the format for CCCD writes.
  uint16_t configValue = *(uint16_t *)pCharData->data;
  char *configValString;

  // Determine what to tell the user
  switch(configValue)
  {
  case GATT_CFG_NO_OPERATION:
    configValString = "Noti/Ind disabled";
    break;
  case GATT_CLIENT_CFG_NOTIFY:
    configValString = "Notifications enabled";
    break;
  case GATT_CLIENT_CFG_INDICATE:
    configValString = "Indications enabled";
    break;
  }

  Display_clearLine(dispHandle, 4);
  Display_print0(dispHandle, 5, 0, "Data Service");
  Display_print0(dispHandle, 7, 0, configValString);

  switch (pCharData->paramID)
  {
    case DS_STREAM_ID:
      // Do something useful with configValue here
      // -------------------------
      Display_print0(dispHandle, 6, 0, "Stream");
      Log_info3("CCCD Change msg: %s %s: %s",
                (IArg)"Data Service",
                (IArg)"Stream",
                (IArg)configValString);

      if (configValue) // 0x0001 and 0x0002 both indicate turned on.
        Clock_start((Clock_Handle)&ds_Stream_clock);
      else
        Clock_stop((Clock_Handle)&ds_Stream_clock);
      break;
  }
}

/*********************************************************************
 * @fn      user_RelayService_ValueChangeDispatchHandler
 *
 * @brief   Handle in user task context a write request sent as message
 *          to the application task.
 *
 * @param   pCharData  pointer to malloc'd char write data
 *
 * @return  None.
 */
void user_RelayService_ValueChangeDispatchHandler(char_data_t *pCharData)
{
  char *data;
  data = Util_convertArrayToHexString(pCharData->data, pCharData->dataLen);

  Display_clearLine(dispHandle, 4);
  Display_print0(dispHandle, 5, 0, "RELAY SERVICE");
  Display_print0(dispHandle, 7, 0, data);

  switch (pCharData->paramID)
  {
    case RS_RELAY1_ID:
      // Do something useful with pCharData->data here
      // -------------------------
      Display_print0(dispHandle, 6, 0, "RELAY1");
      Log_info3("Value Change msg: %s %s: %s",
                (IArg)"RELAY SERVICE",
                (IArg)"RELAY1",
                (IArg)data);
      break;

    case RS_RELAY2_ID:
      // Do something useful with pCharData->data here
      // -------------------------
      Display_print0(dispHandle, 6, 0, "RELAY2");
      Log_info3("Value Change msg: %s %s: %s",
                (IArg)"RELAY SERVICE",
                (IArg)"RELAY2",
                (IArg)data);
      break;

  default:
    return;
  }
}

/*********************************************************************
 * @fn      user_DigitalInputService_CfgChangeDispatchHandler
 *
 * @brief   Handle in user task context a CCCD write request sent as
 *          message to the application task.
 *
 * @param   pCharData  pointer to malloc'd char write data
 *
 * @return  None.
 */
void user_DigitalInputService_CfgChangeDispatchHandler(char_data_t *pCharData)
{
  // Cast received data to uint16, as that's the format for CCCD writes.
  uint16_t configValue = *(uint16_t *)pCharData->data;
  char *configValString;

  // Determine what to tell the user
  switch(configValue)
  {
  case GATT_CFG_NO_OPERATION:
    configValString = "Noti/Ind disabled";
    break;
  case GATT_CLIENT_CFG_NOTIFY:
    configValString = "Notifications enabled";
    break;
  case GATT_CLIENT_CFG_INDICATE:
    configValString = "Indications enabled";
    break;
  }

  Display_clearLine(dispHandle, 4);
  Display_print0(dispHandle, 5, 0, "Digital Input ..");
  Display_print0(dispHandle, 7, 0, configValString);

  switch (pCharData->paramID)
  {
    case DIS_IN1_ID:
      // Do something useful with configValue here
      // -------------------------
      Display_print0(dispHandle, 6, 0, "IN1");
      Log_info3("CCCD Change msg: %s %s: %s",
                (IArg)"Digital Input Service",
                (IArg)"IN1",
                (IArg)configValString);

      if (configValue) // 0x0001 and 0x0002 both indicate turned on.
        Clock_start((Clock_Handle)&dis_IN1_clock);
      else
        Clock_stop((Clock_Handle)&dis_IN1_clock);
      break;
    case DIS_IN2_ID:
      // Do something useful with configValue here
      // -------------------------
      Display_print0(dispHandle, 6, 0, "IN2");
      Log_info3("CCCD Change msg: %s %s: %s",
                (IArg)"Digital Input Service",
                (IArg)"IN2",
                (IArg)configValString);

      if (configValue) // 0x0001 and 0x0002 both indicate turned on.
        Clock_start((Clock_Handle)&dis_IN2_clock);
      else
        Clock_stop((Clock_Handle)&dis_IN2_clock);
      break;
    case DIS_IN3_ID:
      // Do something useful with configValue here
      // -------------------------
      Display_print0(dispHandle, 6, 0, "IN3");
      Log_info3("CCCD Change msg: %s %s: %s",
                (IArg)"Digital Input Service",
                (IArg)"IN3",
                (IArg)configValString);

      if (configValue) // 0x0001 and 0x0002 both indicate turned on.
        Clock_start((Clock_Handle)&dis_IN3_clock);
      else
        Clock_stop((Clock_Handle)&dis_IN3_clock);
      break;
  }
}


#ifdef FEATURE_OAD
/*********************************************************************
 * @fn      SimpleBLEPeripheral_processOadWriteCB
 *
 * @brief   Process a write request to the OAD profile.
 *
 * @param   event      - event type:
 *                       OAD_WRITE_IDENTIFY_REQ
 *                       OAD_WRITE_BLOCK_REQ
 * @param   connHandle - the connection Handle this request is from.
 * @param   pData      - pointer to data for processing and/or storing.
 *
 * @return  None.
 */
void SimpleBLEPeripheral_processOadWriteCB(uint8_t event, uint16_t connHandle,
                                           uint8_t *pData)
{
  oadTargetWrite_t *oadWriteEvt = ICall_malloc( sizeof(oadTargetWrite_t) + \
                                             sizeof(uint8_t) * OAD_PACKET_SIZE);

  if ( oadWriteEvt != NULL )
  {
    oadWriteEvt->event = event;
    oadWriteEvt->connHandle = connHandle;

    oadWriteEvt->pData = (uint8_t *)(&oadWriteEvt->pData + 1);
    memcpy(oadWriteEvt->pData, pData, OAD_PACKET_SIZE);

    Queue_enqueue(hOadQ, (Queue_Elem *)oadWriteEvt);

    // Post the application's semaphore.
    Semaphore_post(sem);
  }
  else
  {
    // Fail silently.
  }
}
#endif //FEATURE_OAD


/*
 *  Callbacks from the Stack Task context
 *****************************************************************************/

/**
 * Callback from GAP Role indicating a role state change.
 */
static void user_gapStateChangeCB(gaprole_States_t newState)
{
  Log_info1("(CB) GAP State change: %d, Sending msg to app.", (IArg)newState);
  user_enqueueRawAppMsg( APP_MSG_GAP_STATE_CHANGE, (uint8_t *)&newState, sizeof(newState) );
}

/**
 * Callback handler for characteristic value changes in generated services, registered with the service.
 */
static void user_service_ValueChangeCB( uint16_t connHandle, uint16_t svcUuid, uint8_t paramID, uint8_t *pValue, uint16_t len )
{
  // See the service header file to compare paramID with characteristic value attribute.
  Log_info2("(CB) Characteristic value change: svc(0x%04x) paramID(%d). Sending msg to app.", (IArg)svcUuid, (IArg)paramID);
  user_enqueueCharDataMsg(APP_MSG_SERVICE_WRITE, connHandle, svcUuid, paramID, pValue, len);
}

/**
 * Callback handler for characteristic configuration changes in generated services, registered with the service.
 */
static void user_service_CfgChangeCB( uint16_t connHandle, uint16_t svcUuid, uint8_t paramID, uint8_t *pValue, uint16_t len )
{
  Log_info2("(CB) Characteristic configuration change: svc(0x%04x) paramID(%d). Sending msg to app.", (IArg)svcUuid, (IArg)paramID);
  user_enqueueCharDataMsg(APP_MSG_SERVICE_CFG, connHandle, svcUuid, paramID, pValue, len);
}

/*
 *  Callbacks from Swi-context
 *****************************************************************************/

/*
 * Handles the Software Interrupt resulting from timeout of the clock object(s)
 * used to demonstrate notifications/indications.
 */
static void user_generic_clockSwiHandler(uint16_t svcUuid, uint16_t paramID)
{
  uint8_t notiData[20];
  uint16_t notiLen = sizeof notiData;

  Log_info2("(SWI) Generic clock handler: svc(0x%04x) paramID(%d)", (IArg)svcUuid, (IArg)paramID);
  static uint8_t someCounter = 0;

  // Get loopback data if char is writable, otherwise fill with junk
  switch(svcUuid) {
    case BUTTON_SERVICE_SERV_UUID:
      switch(paramID) {
        case BS_BUTTON0_ID:
          // Characteristic is not writable, so send something, max 20 bytes (max default ATT MTU size minus header)
          notiLen = MIN(notiLen, BS_BUTTON0_LEN);
          memset(notiData, someCounter++, notiLen);
          break;
        case BS_BUTTON1_ID:
          // Characteristic is not writable, so send something, max 20 bytes (max default ATT MTU size minus header)
          notiLen = MIN(notiLen, BS_BUTTON1_LEN);
          memset(notiData, someCounter++, notiLen);
          break;
        case BS_BUTTON2_T1_ID:
          // Characteristic is not writable, so send something, max 20 bytes (max default ATT MTU size minus header)
          notiLen = MIN(notiLen, BS_BUTTON2_T1_LEN);
          memset(notiData, someCounter++, notiLen);
          break;
        case BS_BUTTON3_T2_ID:
          // Characteristic is not writable, so send something, max 20 bytes (max default ATT MTU size minus header)
          notiLen = MIN(notiLen, BS_BUTTON3_T2_LEN);
          memset(notiData, someCounter++, notiLen);
          break;
      }
      break;
    case DATA_SERVICE_SERV_UUID:
      switch(paramID) {
        case DS_STREAM_ID:
          // Characteristic is writable, so echo back contents.
          DataService_GetParameter( paramID, &notiLen, notiData );
          break;
      }
      break;
    case DIGITAL_INPUT_SERVICE_SERV_UUID:
      switch(paramID) {
        case DIS_IN1_ID:
          // Characteristic is not writable, so send something, max 20 bytes (max default ATT MTU size minus header)
          notiLen = MIN(notiLen, DIS_IN1_LEN);
          memset(notiData, someCounter++, notiLen);
          break;
        case DIS_IN2_ID:
          // Characteristic is not writable, so send something, max 20 bytes (max default ATT MTU size minus header)
          notiLen = MIN(notiLen, DIS_IN2_LEN);
          memset(notiData, someCounter++, notiLen);
          break;
        case DIS_IN3_ID:
          // Characteristic is not writable, so send something, max 20 bytes (max default ATT MTU size minus header)
          notiLen = MIN(notiLen, DIS_IN3_LEN);
          memset(notiData, someCounter++, notiLen);
          break;
      }
      break;
  }

  // Send message to application that it should update the value of the characteristic from Task context.
  user_enqueueCharDataMsg(APP_MSG_SEND_NOTIFICATION, 0xFFFF, svcUuid, paramID, notiData, notiLen);
}

/*
 * Swi handler for clock object(s) used by Button Service.
 * paramID is stored in the clock object for each characteristic.
 */
static void user_ButtonService_clockSwiHandler(UArg paramID)
{
  // Act as a closure for the generic clockSwiHandler, as clock objects only have one parameter associated.
  user_generic_clockSwiHandler(BUTTON_SERVICE_SERV_UUID, paramID);
}

/*
 * Swi handler for clock object(s) used by Data Service.
 * paramID is stored in the clock object for each characteristic.
 */
static void user_DataService_clockSwiHandler(UArg paramID)
{
  // Act as a closure for the generic clockSwiHandler, as clock objects only have one parameter associated.
  user_generic_clockSwiHandler(DATA_SERVICE_SERV_UUID, paramID);
}

/*
 * Swi handler for clock object(s) used by Digital Input Service.
 * paramID is stored in the clock object for each characteristic.
 */
static void user_DigitalInputService_clockSwiHandler(UArg paramID)
{
  // Act as a closure for the generic clockSwiHandler, as clock objects only have one parameter associated.
  user_generic_clockSwiHandler(DIGITAL_INPUT_SERVICE_SERV_UUID, paramID);
}

/******************************************************************************
 *
 *  Utility functions
 *
 *****************************************************************************/
/**
 * Generic message constructor for characteristic data.
 * Sends a message to the application for handling in Task context.
 */
static void user_enqueueCharDataMsg( app_msg_types_t appMsgType, uint16_t connHandle,
                                       uint16_t serviceUUID, uint8_t paramID,
                                       uint8_t *pValue, uint16_t len )
{
  // Called in Stack Task context, so can't do processing here.
  // Send message to application message queue about received data.
  uint16_t readLen = len; // How much data was written to the attribute

  // Allocate memory for the message.
  // Note: The pCharData message doesn't have to contain the data itself, as that's stored in
  //       a variable in the service. However, to prevent data loss if a new value is received
  //       before GetParameter is called, we call GetParameter now.
  app_msg_t *pMsg = ICall_malloc( sizeof(app_msg_t) + sizeof(char_data_t) + readLen );

  if (pMsg != NULL)
  {
    pMsg->type = appMsgType;

    char_data_t *pCharData = (char_data_t *)pMsg->pdu;
    pCharData->svcUUID = serviceUUID; // Use 16-bit part of UUID.
    pCharData->paramID = paramID;
    // Copy data from service now, because it could change before GetParameter is called later.
    memcpy(pCharData->data, pValue, readLen);
    // Update pCharData with how much data we received.
    pCharData->dataLen = readLen;
    // Enqueue the message using pointer to queue node element.
    Queue_enqueue(hApplicationMsgQ, &pMsg->_elem);
    // Let application know there's a message.
    Semaphore_post(sem);
  }
}

/**
 * Generic message constructor for raw data.
 * Sends a message to the application for handling in Task context.
 */
static void user_enqueueRawAppMsg( app_msg_types_t appMsgType, uint8_t *pData, uint16_t len )
{
  // Allocate memory for the message.
  // Note: The pCharData message doesn't have to contain the data itself, as that's stored in
  //       a variable in the service. However, to prevent data loss if a new value is received
  //       before GetParameter is called, we call GetParameter now.
  app_msg_t *pMsg = ICall_malloc( sizeof(app_msg_t) + len );

  if (pMsg != NULL)
  {
    pMsg->type = appMsgType;

    // Copy data from service now, because it could change before GetParameter is called later.
    memcpy(pMsg->pdu, pData, len);

    // Enqueue the message using pointer to queue node element.
    Queue_enqueue(hApplicationMsgQ, &pMsg->_elem);
    // Let application know there's a message.
    Semaphore_post(sem);
  }
}


/**
 * Convenience function for updating characteristic data via char_data_t message
 * MUST run in Task context.
 */
static void user_updateCharVal(char_data_t *pCharData)
{
  switch(pCharData->svcUUID) {
    case LED_SERVICE_SERV_UUID:
      LedService_SetParameter( pCharData->paramID, pCharData->dataLen, pCharData->data );
    break;

    case BUTTON_SERVICE_SERV_UUID:
      ButtonService_SetParameter( pCharData->paramID, pCharData->dataLen, pCharData->data );
    break;

    case DATA_SERVICE_SERV_UUID:
      DataService_SetParameter( pCharData->paramID, pCharData->dataLen, pCharData->data );
    break;

    case RELAY_SERVICE_SERV_UUID:
      RelayService_SetParameter( pCharData->paramID, pCharData->dataLen, pCharData->data );
    break;

    case DIGITAL_INPUT_SERVICE_SERV_UUID:
      DigitalInputService_SetParameter( pCharData->paramID, pCharData->dataLen, pCharData->data );
    break;

  }
}

/*********************************************************************
 * @fn      Util_convertArrayToHexString
 *
 * @brief   Convert {0x01, 0x02} to "01:02" unless array is printable
 *          and it's more than 2 bytes.
 *
 * @param   arr - array
 * @param   len - length of array
 *
 * @return  array as string
 */
char *Util_convertArrayToHexString(uint8_t *arr, uint8_t len)
{
  char        hex[] = "0123456789ABCDEF";
  static char str[17];
  char        *pStr = str;
  uint8_t     avail = 16;

  memset(str, 0, avail);

  // Don't convert to hex if all chars printable (A-Za-z0-9SP)
  uint8_t i;
  for (i = 0; i < avail && len > 2; ++i)
  {
    if ( !((arr[i] >= 'a' && arr[i] <= 'z') ||
           (arr[i] >= 'A' && arr[i] <= 'Z') ||
           (arr[i] >= '0' && arr[i] <= '9') ||
           (arr[i] == ' ')) )
      { break; }
  }
  if (i == MIN(len, avail))
  {
      memcpy(str, arr, i);
  }
  else // It's not all printable characters. Convert to hexadecimal.
  {
      if (len)
      {
        *pStr++ = hex[*arr >> 4];
        *pStr++ = hex[*arr++ & 0x0F];
        avail -= 2;
        len--;
      }
      while (len && avail > 3)
      {
        if (avail) { *pStr++ = ':'; avail -= 1; };
        *pStr++ = hex[*arr >> 4];
        *pStr++ = hex[*arr++ & 0x0F];
        avail -= 2;
        len--;
      }

      if (len && avail)
        *pStr++ = ':'; // Indicate not all data fit on line.
  }

  return str;
}


/*********************************************************************
*********************************************************************/
